<?php

namespace App\Models\Users;

use App\Core\ParamConverter\JsonModelConverter;
use App\Entity\Agreement\Agreement;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;

class UserModel implements JsonModelConverter
{
    const ID = 'id';
    const NAME = 'name';
    const SURNAME = 'surname';
    const EMAIL = 'email';
    const PASSWORD = 'password';
    const REGULATIONS = 'regulations';
    const COMMERCIAL_INFORMATION = 'commercial_information';

    const REGISTRATION = [self::NAME, self::SURNAME, self::EMAIL, self::PASSWORD, self::REGULATIONS, self::COMMERCIAL_INFORMATION];
    const PERSONAL_DATA = [self::NAME, self::SURNAME];

    #[Assert\NotBlank(groups: [self::NAME])]
    #[Assert\Type(type: 'string', groups: [self::NAME])]
    #[Assert\Length(max: 50, groups: [self::NAME])]
    private string $name;

    #[Assert\NotBlank(groups: [self::SURNAME])]
    #[Assert\Type(type: 'string', groups: [self::SURNAME])]
    #[Assert\Length(max: 50, groups: [self::SURNAME])]
    private string $surname;

    #[Assert\NotBlank(groups: [self::EMAIL])]
    #[Assert\Email(groups: [self::EMAIL])]
    private string $email;

    #[Assert\NotBlank(groups: [self::PASSWORD])]
    #[Assert\Length(min: 8, max: 255, groups: [self::PASSWORD])]
    #[Assert\Regex(
        pattern: '/(?=.*[a-z])(?=.*[A-Z])(?=.*[0-9])(?=.*[^A-Za-z0-9])/',
        message: 'Password must contain at least one uppercase, one lowercase, one special character, and one number',
        groups: [self::PASSWORD]
    )]
    private string $password;

    #[Assert\IsTrue(groups: [self::REGULATIONS])]
    #[Assert\Type(type: 'bool', groups: [self::REGULATIONS])]
    private bool $isRegulations;

    #[Assert\NotNull(groups: [self::COMMERCIAL_INFORMATION])]
    #[Assert\Type(type: 'bool', groups: [self::COMMERCIAL_INFORMATION])]
    private bool $isCommercialInformation;

    /** @return string */
    #[Groups([self::NAME])]
    public function getName(): string
    {
        return $this->name;
    }

    /** @param string $name */
    public function setName(string $name): void
    {
        $this->name = $name;
    }

    /** @return string */
    #[Groups([self::SURNAME])]
    public function getSurname(): string
    {
        return $this->surname;
    }

    /** @param string $surname */
    public function setSurname(string $surname): void
    {
        $this->surname = $surname;
    }

    /** @return string */
    #[Groups([self::EMAIL])]
    public function getEmail(): string
    {
        return $this->email;
    }

    /** @param string $email */
    public function setEmail(string $email): void
    {
        $this->email = $email;
    }

    /** @return string */
    #[Groups([self::PASSWORD])]
    public function getPassword(): string
    {
        return $this->password;
    }

    /** @param string $password */
    public function setPassword(string $password): void
    {
        $this->password = $password;
    }

    /** @return bool */
    #[Groups([self::REGULATIONS])]
    public function getIsRegulations(): bool
    {
        return $this->isRegulations;
    }

    /** @param bool $isRegulations */
    public function setIsRegulations(bool $isRegulations): void
    {
        $this->isRegulations = $isRegulations;
    }

    /** @return bool */
    #[Groups([self::COMMERCIAL_INFORMATION])]
    public function getIsCommercialInformation(): bool
    {
        return $this->isCommercialInformation;
    }

    /** @param bool $isCommercialInformation */
    public function setIsCommercialInformation(bool $isCommercialInformation): void
    {
        $this->isCommercialInformation = $isCommercialInformation;
    }

    public function getAgreementOriginalNames(): array
    {
        return [
            $this->isCommercialInformation ? Agreement::COMMERCIAL_INFORMATION : null
        ];
    }
}