<?php

namespace App\Models\Units;

use App\Core\File\FileToUpload\ImageToUpload;
use App\Core\ParamConverter\FormDataModelConverter;
use App\Entity\Units\Helper\Type;
use App\Entity\Units\UnitID;
use Doctrine\Common\Collections\ArrayCollection;
use Symfony\Component\HttpFoundation\File\UploadedFile;
use Symfony\Component\Serializer\Annotation\Groups;
use Symfony\Component\Validator\Constraints as Assert;

class UnitModel implements FormDataModelConverter
{
    const CREATE = 'create';
    const UPDATE = 'update';

    /** @var string|null $unitType */
    #[Assert\NotBlank(groups: [self::CREATE])]
    #[Assert\Type(type: 'string', groups: [self::CREATE])]
    #[Assert\Choice(
        choices: Type::CHOICES,
        groups: [self::CREATE]
    )]
    private ?string $unitType;

    #[Assert\NotBlank(groups: [self::CREATE, self::UPDATE])]
    #[Assert\Type(type: 'string', groups: [self::CREATE, self::UPDATE])]
    #[Assert\Length(max: 100, groups: [self::CREATE, self::UPDATE])]
    private string $title;

    #[Assert\NotBlank(groups: [self::CREATE])]
    #[Assert\Valid]
    private ?ImageToUpload $photo;

    #[Assert\NotBlank(groups: [self::CREATE])]
    #[Assert\Valid]
    private ?ImageToUpload $banner;

    #[Assert\NotBlank(groups: [self::CREATE, self::UPDATE])]
    #[Assert\Type(type: 'string', groups: [self::CREATE, self::UPDATE])]
    #[Assert\Length(max: 200, groups: [self::CREATE, self::UPDATE])]
    private string $description;

    #[Assert\NotBlank(groups: [self::CREATE, self::UPDATE])]
    #[Assert\Type(type: 'string', groups: [self::CREATE, self::UPDATE])]
    private string $unitChildren;

    /**
     * @return Type
     */
    #[Groups([self::CREATE])]
    public function getUnitType(): Type
    {
        return new Type($this->unitType);
    }

    /**
     * @param string|null $unitType
     */
    public function setUnitType(?string $unitType): void
    {
        $this->unitType = $unitType;
    }

    /**
     * @return string
     */
    #[Groups([self::CREATE, self::UPDATE])]
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @param string $title
     */
    public function setTitle(string $title): void
    {
        $this->title = $title;
    }

    /**
     * @return ImageToUpload
     */
    public function getPhoto(): ImageToUpload
    {
        return $this->photo;
    }

    /**
     * @return bool
     */
    public function isPhoto(): bool
    {
        return !empty($this->photo);
    }

    /**
     * @param UploadedFile|string $photo
     */
    public function setPhoto(UploadedFile|string $photo): void
    {
        $this->photo = !is_string($photo) ? new ImageToUpload($photo) : null;
    }

    /**
     * @return ImageToUpload
     */
    public function getBanner(): ImageToUpload
    {
        return $this->banner;
    }

    /**
     * @return bool
     */
    public function isBanner(): bool
    {
        return !empty($this->banner);
    }

    /**
     * @param UploadedFile|string $banner
     */
    public function setBanner(UploadedFile|string $banner): void
    {
        $this->banner = !is_string($banner) ? new ImageToUpload($banner) : null;
    }

    /**
     * @return string
     */
    #[Groups([self::CREATE, self::UPDATE])]
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @param string $description
     */
    public function setDescription(string $description): void
    {
        $this->description = $description;
    }

    /**
     * @param string $unitChildren
     */
    public function setUnitChildren(string $unitChildren): void
    {
        $this->unitChildren = $unitChildren;
    }

    /**
     * @return UnitID[]
     */
    #[Groups([self::CREATE, self::UPDATE])]
    public function getUnitChildren(): array
    {
        return (new ArrayCollection(json_decode($this->unitChildren)))
            ->map(fn($unitChild) => UnitID::fromString($unitChild))
            ->toArray();
    }
}