<?php

namespace App\Models\Reports;

use App\Core\ParamConverter\JsonModelConverter;
use App\Core\Validator as CustomAssert;
use App\Entity\Posts\Post;
use App\Entity\Posts\PostID;
use App\Entity\Reports\Helper\ReportName;
use App\Entity\Units\Unit;
use App\Entity\Units\UnitID;
use DateTimeImmutable;
use Exception;
use OpenApi\Annotations as OA;
use Symfony\Component\Serializer\Annotation\Ignore;
use Symfony\Component\Validator\Constraints as Assert;

class ReportModel implements JsonModelConverter
{
    #[Assert\NotBlank]
    #[CustomAssert\EntityID(entityClass: Unit::class)]
    private string $unitId;

    /** @var string $dateStart */
    #[Assert\NotBlank]
    #[Assert\Type(type: 'string')]
    #[Assert\Date]
    private string $dateStart;

    /** @var string $dateEnd */
    #[Assert\NotBlank]
    #[Assert\Type(type: 'string')]
    #[Assert\Date]
    private string $dateEnd;

    #[Assert\NotBlank]
    #[CustomAssert\EntityID(entityClass: Post::class)]
    private string $postId;

    #[Assert\NotBlank]
    #[Assert\Length(max: 300)]
    private string $description;

    #[Ignore]
    #[Assert\Choice(choices: ReportName::NAMES)]
    private string $reportName;

    /**
     * @param string $unitId
     * @param string $dateStart
     * @param string $dateEnd
     * @param string $postId
     * @param string $description
     * @param string $reportName
     */
    public function __construct(string $unitId, string $dateStart, string $dateEnd, string $postId, string $description, string $reportName = ReportName::REPORT_QUARTER)
    {
        $this->unitId = $unitId;
        $this->dateStart = $dateStart;
        $this->dateEnd = $dateEnd;
        $this->postId = $postId;
        $this->description = $description;
        $this->reportName = $reportName;
    }

    /**
     * @OA\Property(type="string")
     */
    public function getUnitId()
    {
        return UnitID::fromString($this->unitId);
    }

    /**
     * @return string
     */
    public function getDateStart(): string
    {
        return $this->dateStart;
    }

    /**
     * @return string
     */
    public function getDateEnd(): string
    {
        return $this->dateEnd;
    }


    /**
     * @return DateTimeImmutable
     * @throws Exception
     */
    #[Ignore]
    public function getDateStartObject(): DateTimeImmutable
    {
        return new DateTimeImmutable($this->dateStart);
    }

    /**
     * @return DateTimeImmutable
     * @throws Exception
     */
    #[Ignore]
    public function getDateEndObject(): DateTimeImmutable
    {
        return new DateTimeImmutable($this->dateEnd);
    }

    /**
     * @OA\Property(type="string")
     */
    public function getPostId()
    {
        return PostID::fromString($this->postId);
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @Ignore
     * @return string
     */
    public function getReportName(): string
    {
        return $this->reportName;
    }
}