<?php

namespace App\Entity\Units;

use App\Entity\Contents\Content;
use App\Entity\HearthAdoption\HearthAdoption;
use App\Entity\Purposes\Purpose;
use App\Entity\Units\Helper\Type;
use App\Entity\Users\User;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Symfony\Component\Uid\Uuid;
use Doctrine\ORM\Mapping as ORM;
use App\Entity\Images\Image;

/**
 * @ORM\Entity()
 */
class Unit
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private UnitID|Uuid $uuid;

    /**
     * @ORM\Column(type="string", length=100)
     */
    private string $title;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $description;

    /**
     * @ORM\OneToOne(targetEntity=Image::class, cascade={"persist"})
     * @ORM\JoinColumn(name="photo_id", referencedColumnName="uuid", nullable=false)
     */
    private Image $photo;

    /**
     * @ORM\OneToOne(targetEntity=Image::class, cascade={"persist"})
     * @ORM\JoinColumn(name="banner_id", referencedColumnName="uuid", nullable=false)
     */
    private Image $banner;

    /**
     * @ORM\OneToOne(targetEntity=Content::class, cascade={"persist"})
     * @ORM\JoinColumn(name="card_id", referencedColumnName="uuid", nullable=false)
     */
    private Content $cardInfo;

    /**
     * @ORM\OneToOne(targetEntity=Purpose::class, cascade={"persist"})
     * @ORM\JoinColumn(name="purpose_id", referencedColumnName="uuid", nullable=false)
     */
    private Purpose $purpose;

    /**
     * @ORM\OneToOne(targetEntity=HearthAdoption::class, cascade={"persist"})
     * @ORM\JoinColumn(name="adoption_id", referencedColumnName="uuid", nullable=true)
     */
    private ?HearthAdoption $hearthAdoption;

    /**
     * @ORM\Column(type="string", length=1)
     */
    private string $type;

    /**
     * @ORM\Column(type="boolean", options={"default": 0})
     */
    private bool $isActive = false;

    /**
     * @ORM\Column(type="datetime_immutable", nullable=true)
     */
    private ?\DateTimeImmutable $publishedAt;

    /**
     * @ORM\Column(type="datetime_immutable", options={"default": "CURRENT_TIMESTAMP"})
     */
    private \DateTimeImmutable $createdAt;


    /**
     * @ORM\ManyToMany(targetEntity=Unit::class, mappedBy="unitChildren")
     */
    private Collection $unitParents;

    /**
     * @ORM\JoinTable(name="unit_children",
     *     joinColumns={@ORM\JoinColumn(name="parent_unit_id", referencedColumnName="uuid")},
     *     inverseJoinColumns={@ORM\JoinColumn(name="children_unit_id", referencedColumnName="uuid")}
     * )
     * @ORM\ManyToMany(targetEntity=Unit::class, inversedBy="unitParents")
     */
    private Collection $unitChildren;

    /**
     * @param string $title
     * @param Image $banner
     * @param Image $photo
     * @param string $description
     * @param Content $cardInfo
     * @param Purpose $purpose
     * @param Type $type
     */
    public function __construct(string $title, Image $banner, Image $photo, string $description, Content $cardInfo, Purpose $purpose, Type $type)
    {
        $this->uuid = UnitID::generate();
        $this->title = $title;
        $this->banner = $banner;
        $this->photo = $photo;
        $this->description = $description;
        $this->cardInfo = $cardInfo;
        $this->purpose = $purpose;
        $this->type = $type;
        $this->createdAt = new \DateTimeImmutable();
        $this->unitChildren = new ArrayCollection();
        $this->unitParents = new ArrayCollection();
    }

    /**
     * @return UnitID
     */
    public function getUuid(): UnitID
    {
        return UnitID::fromString($this->uuid);
    }

    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }

    /**
     * @param string $title
     */
    public function updateTitle(string $title): void
    {
        $this->title = $title;
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @param string $description
     */
    public function updateDescription(string $description): void
    {
        $this->description = $description;
    }

    /**
     * @return Image|null
     */
    public function getPhoto(): ?Image
    {
        return $this->photo;
    }

    /**
     * @return Image|null
     */
    public function getBanner(): ?Image
    {
        return $this->banner;
    }

    /**
     * @return Content
     */
    public function getCardInfo(): Content
    {
        return $this->cardInfo;
    }

    /**
     * @return Purpose
     */
    public function getPurpose(): Purpose
    {
        return $this->purpose;
    }

    /**
     * @return HearthAdoption|null
     */
    public function getHearthAdoption(): ?HearthAdoption
    {
        return $this->hearthAdoption;
    }

    /**
     * @param int $childrenCount
     * @param int $monthlyCost
     */
    public function updateHearthAdoptionData(int $childrenCount, int $monthlyCost): void
    {
        $this->hearthAdoption->updateChildren($childrenCount);
        $this->hearthAdoption->updateMonthlyCost($monthlyCost);
        $this->hearthAdoption->updateModifiedAt();
    }

    /**
     * @param string $content
     */
    public function updateHearthAdoptionContent(string $content): void
    {
        $this->hearthAdoption
            ->getContent()
            ->updateContent($content);
    }

    /**
     * @param HearthAdoption $hearthAdoption
     */
    public function setHearthAdoption(HearthAdoption $hearthAdoption): void
    {
        $this->hearthAdoption = $hearthAdoption;
    }

    /**
     * @return Type
     */
    public function getType(): Type
    {
        return new Type($this->type);
    }

    /**
     * @param Image|null $photo
     */
    public function setPhoto(?Image $photo): void
    {
        $this->photo = $photo;
    }

    /**
     * @param Image|null $banner
     */
    public function setBanner(?Image $banner): void
    {
        $this->banner = $banner;
    }

    /**
     * @return bool
     */
    public function isActive(): bool
    {
        return $this->isActive;
    }

    public function isUnpublish(): bool
    {
        return !$this->isActive;
    }

    public function publish(): void
    {
        $this->isActive = true;
        $this->publishedAt = new \DateTimeImmutable();
    }

    public function unpublish(): void
    {
        $this->isActive = false;
        $this->publishedAt = null;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getPublishedAt(): ?\DateTimeImmutable
    {
        return $this->publishedAt;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getCreatedAt(): \DateTimeImmutable
    {
        return $this->createdAt;
    }

    /**
     * @return Collection|Unit[]
     */
    public function getUnitChildren(): Collection
    {
        return $this->unitChildren;
    }

    public function getUnitChildrenIDs(): array
    {
        return $this->unitChildren
            ->map(fn($agreement) => $agreement->getUuid())
            ->toArray();
    }

    public function addUnitChildren(Unit $unit): self
    {
        if (!$this->unitChildren->contains($unit)) {
            $this->unitChildren[] = $unit;
        }
        return $this;
    }

    public function removeUnitChildren(Unit $unit): self
    {
        $this->unitChildren->removeElement($unit);

        return $this;
    }

    /**
     * @return Collection|Unit[]
     */
    public function getUnitParents(): Collection
    {
        return $this->unitParents;
    }

    public function addUnitParents(Unit $unit): self
    {
        if (!$this->unitParents->contains($unit)) {
            $this->unitParents[] = $unit;
        }
        return $this;
    }

}