<?php

namespace App\Entity\Reports;

use App\Entity\Posts\Post;
use App\Entity\Units\Unit;
use DateTimeImmutable;
use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;

/**
 * @ORM\Entity()
 */
class Report
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private ReportID|Uuid $uuid;

    /**
     * @ORM\Column(type="string", length=1)
     */
    private string $name;

    /**
     * @ORM\ManyToOne(targetEntity=Unit::class)
     * @ORM\JoinColumn(name="unit_id", referencedColumnName="uuid", nullable=false)
     */
    private Unit $unit;

    /**
     * @ORM\ManyToOne(targetEntity=Post::class)
     * @ORM\JoinColumn(name="post_id", referencedColumnName="uuid", nullable=false)
     */
    private Post $post;

    /**
     * @ORM\Column(type="datetime_immutable", options={"default": "CURRENT_TIMESTAMP"})
     */
    private DateTimeImmutable $dateStart;

    /**
     * @ORM\Column(type="datetime_immutable", options={"default": "CURRENT_TIMESTAMP"})
     */
    private DateTimeImmutable $dateEnd;

    /**
     * @ORM\Column(type="string", length=300)
     */
    private string $description;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private DateTimeImmutable $createdAt;

    /**
     * @param string $name
     * @param Unit $unit
     * @param Post $post
     * @param DateTimeImmutable $dateStart
     * @param DateTimeImmutable $dateEnd
     * @param string $description
     */
    public function __construct(
        string $name,
        Unit $unit,
        Post $post,
        DateTimeImmutable $dateStart,
        DateTimeImmutable $dateEnd,
        string $description
    )
    {
        $this->uuid = ReportID::generate();
        $this->name = $name;
        $this->unit = $unit;
        $this->post = $post;
        $this->dateStart = $dateStart;
        $this->dateEnd = $dateEnd;
        $this->description = $description;
        $this->createdAt = new DateTimeImmutable();
    }

    /**
     * @return ReportID
     */
    public function getUuid(): ReportID
    {
        return ReportID::fromString($this->uuid);
    }

    /**
     * @return string
     */
    public function getName(): string
    {
        return $this->name;
    }

    /**
     * @return Unit
     */
    public function getUnit(): Unit
    {
        return $this->unit;
    }

    /**
     * @return Post
     */
    public function getPost(): Post
    {
        return $this->post;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getDateStart(): DateTimeImmutable
    {
        return $this->dateStart;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getDateEnd(): DateTimeImmutable
    {
        return $this->dateEnd;
    }

    /**
     * @return string
     */
    public function getDescription(): string
    {
        return $this->description;
    }

    /**
     * @return DateTimeImmutable
     */
    public function getCreatedAt(): DateTimeImmutable
    {
        return $this->createdAt;
    }
}