<?php

namespace App\Controller\Users\Data;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\ObjectValue\PaginationResults;
use App\Entity\Permissions\Permission;
use App\Entity\Users\ReadModel\VolunteersBalanceQueryInterface;
use App\Entity\Users\User;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\Normalizer\AbstractNormalizer;

/** @OA\Tag(name="Volunteers") */
#[Route('/api/volunteers', name: 'volunteers_')]
class VolunteerBalanceDataController extends AbstractController
{
    public function __construct(
        private VolunteersBalanceQueryInterface $volunteersBalanceQuery,
        private PaginationResults $paginationResults,
    )
    {
    }

    /**
     * @OA\Get(summary="Show voluteer balance history")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{volunteerId}/balance-history', name: 'show_balance_history', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['volunteerId' => 'uuid']])]
    public function showBalanceHistory(Request $request, User $user): JsonResponse
    {
        $results = $this->volunteersBalanceQuery->getVolunteerBalanceHistoryByUserID($user->getUuid(), PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /**
     * @OA\Get(summary="Show voluteer balance")
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{volunteerId}/balance', name: 'show_balance', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['volunteerId' => 'uuid']])]
    public function showBalance(User $user): JsonResponse
    {
        return $this->json(
            $this->volunteersBalanceQuery->getVolunteerBalanceByUserID($user->getUuid())
        );
    }

    /**
     * @OA\Get(summary="Show voluteers list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[Route(path: '', name: 'list', methods: ['GET'])]
    public function list(Request $request): JsonResponse
    {
        $results = $this->volunteersBalanceQuery->getVolunteers(PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /** @OA\Get(summary="Find volunteer by reference code") */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/{referenceCode}', name: 'exist', methods: ['GET'])]
    public function volunteerExist(string $referenceCode): JsonResponse
    {
        $results = $this->volunteersBalanceQuery->getVolunteerByReferenceCode($referenceCode);
        return $this->json(
            $results, context: [AbstractNormalizer::IGNORED_ATTRIBUTES => ['totalCount', 'adminOperationsCount', 'paymentOperationsCount']]
        );
    }
}