<?php

namespace App\Controller\Users\Data;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\ObjectValue\PaginationResults;
use App\Entity\Orders\ReadModel\OrdersQueryInterface;
use App\Entity\Permissions\Permission;
use App\Entity\Users\User;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Users")
 */
#[Route('/api/users', name: 'users_')]
class UserOrderDataController extends AbstractController
{
    public function __construct(
        private OrdersQueryInterface $ordersQuery,
        private PaginationResults $paginationResults
    ) {}

    /**
     * @OA\Get(summary="User payment history list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/payment-history', name: 'payment_history_list', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    public function paymentHistoryList(User $user, Request $request): JsonResponse
    {
        $results = $this->ordersQuery->findAllPaymentByUserId($user->getUuid(), PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }

    /**
     * @OA\Get(summary="User subscription list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::ACCOUNT_DATA_MANAGEMENT)]
    #[IsGranted(Permission::PRIVATE_ACCESS, 'user')]
    #[Route(path: '/{userId}/subscriptions', name: 'subscription_list', methods: ['GET'])]
    #[ParamConverter('user', options: ['mapping' => ['userId' => 'uuid']])]
    public function subscriptionList(User $user, Request $request): JsonResponse
    {
        $results = $this->ordersQuery->findAllSubscriptionByUserId($user->getUuid(), PaginationRequest::build($request));
        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }
}