<?php

namespace App\Controller\Units\Data;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Entity\Permissions\Permission;
use App\Core\Paginator\ObjectValue\PaginationResults;
use App\Entity\Posts\ReadModel\PostsQueryInterface;
use App\Entity\Units\UnitID;
use App\Entity\Users\User;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Routing\Annotation\Route;
use OpenApi\Annotations as OA;

/**
 * @OA\Tag(name="Units")
 */
#[Route(path: '/api/units', name: 'units_')]
class UnitPostsDataController extends AbstractController
{
    public function __construct(
        private PostsQueryInterface $postsQuery,
        private PaginationResults $paginationResults
    )
    {
    }

    /**
     * @OA\Get(summary="Post list")
     * @OA\Parameter(name="page", in="query", description="page number")
     * @OA\Parameter(name="perPage", in="query", description="item per page")
     */
    #[IsGranted(Permission::PUBLIC_ACCESS)]
    #[Route(path: '/{unitId}/posts', name: 'post_list', methods: ['GET'])]
    public function postsList(string $unitId, Request $request): JsonResponse
    {
        /** @var User|null $user */
        $user = $this->getUser();
        $unitId = UnitID::fromString($unitId);
        $paginationRequest = PaginationRequest::build($request);

        $results = $user && $user->hasUnitAdministrationPermission() || $user->hasUnitRedactionPermission() ?
            $this->postsQuery->findAllPostsByUnitId($unitId, $paginationRequest)
            :
            $this->postsQuery->findPublishedPostsByUnitId($unitId, $paginationRequest);

        return $this->json(
            $this->paginationResults->data($results)->totalRow()
        );
    }
}