<?php

namespace App\Controller\Orders;

use App\Core\ValidatorModel\ValidatorModelInterface;
use App\Entity\Permissions\Permission;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use App\Models\Orders\OrderHearthAdoptionModel;
use App\Models\Orders\OrderPurposeModel;
use App\Services\Orders\CreateOrderForCyclicHearthAdoption;
use App\Services\Orders\CreateOrderForOneTimeHearthAdoption;
use OpenApi\Annotations as OA;
use Nelmio\ApiDocBundle\Annotation\Model;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * @OA\Tag(name="Orders subscription")
 */
#[Route(path: '/api/orders', name: 'orders_')]
class HearthAdoptionOrderController extends AbstractController
{
    public function __construct(
        private SerializerInterface $serializer,
        private ValidatorModelInterface $validatorModel,
        private CreateOrderForCyclicHearthAdoption $createOrderForCyclicHearthAdoption,
        private CreateOrderForOneTimeHearthAdoption $createOrderForOneTimeHearthAdoption
    )
    {
    }

    /**
     * @OA\Post(summary="Make new payments for unit purpose")
     * @OA\RequestBody(
     *     @Model(type=OrderHearthAdoptionModel::class, groups={OrderHearthAdoptionModel::CHILDREN_REQUIRED})
     * )
     * @OA\Response(response=200, description="Return url to redirect")
     * @param Request $request
     * @param Unit $unit
     * @return Response
     */
    #[IsGranted(Permission::HEART_ADOPTION_PAYMENT)]
    #[Route('/{unitID}/new-cyclic-subscription', name: 'new_cyclic_subscription', methods: ['POST'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitID' => 'uuid']])]
    public function cyclicSubscription(Request $request, Unit $unit): Response
    {
        /** @var User|null $user */
        $user = $this->getUser();
        $orderHearthAdoptionModel = $this->prepareModel($request, $user, [OrderHearthAdoptionModel::CHILDREN_REQUIRED]);

        $redirectUrl = $this->createOrderForCyclicHearthAdoption->hearthAdoptionCyclicSubscription($orderHearthAdoptionModel, $unit);

        return $this->json(['redirectURL' => $redirectUrl]);
    }

    /**
     * @OA\Post(summary="Make new payments for unit purpose")
     * @OA\RequestBody(
     *     @Model(type=OrderHearthAdoptionModel::class, groups={OrderHearthAdoptionModel::CHILDREN_REQUIRED, OrderHearthAdoptionModel::MONTHS_REQUIRED})
     * )
     * @OA\Response(response=200, description="Return url to redirect")
     * @param Request $request
     * @param Unit $unit
     * @return Response
     */
    #[IsGranted(Permission::HEART_ADOPTION_PAYMENT)]
    #[Route('/{unitID}/new-one-time-subscription', name: 'new_one_time_subscription', methods: ['POST'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitID' => 'uuid']])]
    public function oneTimeSubscription(Request $request, Unit $unit): Response
    {
        /** @var User|null $user */
        $user = $this->getUser();
        $orderHearthAdoptionModel = $this->prepareModel($request, $user, [OrderHearthAdoptionModel::CHILDREN_REQUIRED, OrderHearthAdoptionModel::MONTHS_REQUIRED]);

        $redirectUrl = $this->createOrderForOneTimeHearthAdoption->hearthAdoptionOneTimeSubscription($orderHearthAdoptionModel, $unit);

        return $this->json(['redirectURL' => $redirectUrl]);
    }

    private function prepareModel(Request $request, User $user, array $validationGroups): OrderHearthAdoptionModel
    {
        /** @var OrderHearthAdoptionModel $orderHearthAdoptionModel */
        $orderHearthAdoptionModel = $this->serializer->deserialize($request->getContent(), OrderHearthAdoptionModel::class, 'json');
        $orderHearthAdoptionModel->setUser($user);

        $this->validatorModel->validate(value: $orderHearthAdoptionModel, groups: $validationGroups);

        return $orderHearthAdoptionModel;
    }
}