<?php

namespace App\Adapter\Units\ReadModel;

use App\Core\Doctrine\SubQueryBuilder;
use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\Orders\ReadModel\OrdersSubQueryInterface;
use App\Entity\Units\Helper\Type;
use App\Entity\Units\ReadModel\UnitChildrenDTO;
use App\Entity\Units\ReadModel\UnitDataDTO;
use App\Entity\Units\ReadModel\UnitDetailsDTO;
use App\Entity\Units\ReadModel\UnitAdminListDTO;
use App\Entity\Units\ReadModel\UnitListDetailsDTO;
use App\Entity\Units\ReadModel\UnitsQueryInterface;
use App\Entity\Units\ReadModel\UnitToAssignListDTO;
use App\Entity\Units\UnitID;
use App\Entity\Users\UserID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Query\QueryBuilder;
use Doctrine\DBAL\Driver\Exception as DriverException;

class UnitsQuery implements UnitsQueryInterface
{
    public function __construct(
        private Connection              $connection,
        private OrdersSubQueryInterface $ordersSubQuery,
        private PaginatorInterface      $paginator
    )
    {
    }

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllActiveUnits(PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('u.uuid as unitId')
            ->addSelect('i.image as photo')
            ->addSelect('u.title')
            ->addSelect('u.description')
            ->addSelect('u.type')
            ->addSelect('u.is_active as isActive')
            ->addSelect('p.money as moneyPurpose')
            ->addSelect('a.monthly_cost as monthlyCost')
            ->addSelect('(COALESCE(sO.collectedMoney,0) + COALESCE(uC.collectedMoney,0)) as collectedMoney')
            ->addSelect('a.children')
            ->addSelect('s.supportedChildren as supportedChildren')
            ->from('unit', 'u')
            ->where('u.is_active = :status')
            ->leftJoin('u', 'image', 'i', 'u.photo_id = i.uuid')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('u', 'hearth_adoption', 'a', 'u.adoption_id = a.uuid')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyQuery(), $qb), 'sO', 'u.uuid = sO.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->supportedChildrenQuery(), $qb), 's', 'u.uuid = s.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->collectedMoneyInUnitChildren(), $qb), 'uC', 'u.uuid = uC.unit_id')
            ->groupBy('u.uuid')
            ->addGroupBy('collectedMoney')
            ->addGroupBy('supportedChildren')
            ->addOrderBy('u.published_at', 'DESC')
            ->setParameter('status', true);

        $this->paginator->addPagination($query, $paginationRequest);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => UnitListDetailsDTO::fromArray($data), $results);
    }


    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllUnits(PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();
        $query = $qb
            ->select('u.uuid as unitId')
            ->addSelect('u.title as unitTitle')
            ->addSelect('u.type')
            ->addSelect('a.children')
            ->addSelect('a.monthly_cost as monthlyCost')
            ->addSelect('p.title as purposeTitle')
            ->addSelect('p.money as moneyPurpose')
            ->addSelect('p.started_at as startedAt')
            ->addSelect('u.is_active as isPublish')
            ->addSelect('u.published_at as publishedAt')
            ->from('unit', 'u')
            ->leftJoin('u', 'hearth_adoption', 'a', 'u.adoption_id = a.uuid')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->orderBy('u.type', 'ASC')
            ->addOrderBy('u.created_at', 'DESC');

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => UnitAdminListDTO::fromArray($data), $results);
    }

    /**
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findUnitDataById(UnitID $unitID, ?UserID $userID): ?UnitDataDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('u.title')
            ->addSelect('pI.image as photo')
            ->addSelect('bI.image as banner')
            ->addSelect('u.type')
            ->addSelect('p.title as purposeTitle')
            ->addSelect('p.money as moneyPurpose')
            ->addSelect('(COALESCE(sO.collectedMoney,0) + COALESCE(uC.collectedMoney,0)) as collectedMoney')
            ->addSelect('a.children')
            ->addSelect('s.supportedChildren as supportedChildren')
            ->addSelect('a.monthly_cost as monthlyCost')
            ->addSelect('rP.user_id AS userId')
            ->from('unit', 'u')
            ->leftJoin('u', 'image', 'pI', 'u.photo_id = pI.uuid')
            ->leftJoin('u', 'image', 'bI', 'u.banner_id = bI.uuid')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyQuery(), $qb), 'sO', 'u.uuid = sO.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->supportedChildrenQuery(), $qb), 's', 'u.uuid = s.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->collectedMoneyInUnitChildren(), $qb), 'uC', 'u.uuid = uC.unit_id')
            ->leftJoin('u', 'hearth_adoption', 'a', 'u.adoption_id = a.uuid')
            ->leftJoin('u', 'redactor_partners', 'rP', 'u.uuid = rP.unit_id AND rP.user_id = :userId')
            ->where('u.uuid = :unitId')
            ->setParameter('unitId', $unitID, 'uuid')
            ->setParameter('userId', $userID, 'uuid')
            ->execute()
            ->fetch();

        return $result ? UnitDataDTO::fromArray($result) : null;
    }

    /**
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findUnitDetailsByUnitId(UnitID $unitID): ?UnitDetailsDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('u.type')
            ->addSelect('u.title')
            ->addSelect('pI.image as photo')
            ->addSelect('bI.image as banner')
            ->addSelect('u.description')
            ->from('unit', 'u')
            ->where('u.uuid = :unitId')
            ->leftJoin('u', 'image', 'pI', 'u.photo_id = pI.uuid')
            ->leftJoin('u', 'image', 'bI', 'u.banner_id = bI.uuid')
            ->setParameter('unitId', $unitID, 'uuid')
            ->execute()
            ->fetch();

        $result['hasParent'] = $this->unitHasParents($unitID);

        return $result ? UnitDetailsDTO::fromArray($result) : null;
    }

    public function findAllPossibilityChildren(UnitID $unitID = null): array
    {
        if ($unitID !== null && !$this->unitIsEmergency($unitID)) {
            return [];
        }

        if ($unitID !== null && $this->unitHasParents($unitID)) {
            return [];
        }

        $qb = $this->connection->createQueryBuilder();

        $unitChildrenUUIDs = [];
        $unitChildren = [];

        if ($unitID !== null)
        {
            $unitChildren = $this->findAllUnitChildrenByUnitID($unitID);
            $unitChildrenUUIDs = array_map(fn($child) => $child['unitId'], $unitChildren);
        }

        $qb
            ->select('u.title AS unitTitle, u.uuid AS unitId, uC.parent_unit_id AS parentUnitId')
            ->from('unit', 'u')
            ->leftJoin('u', 'unit_children', 'uC', 'uC.parent_unit_id = u.uuid')
            ->where('uC.children_unit_id IS NULL')
            ->andWhere('u.type = :type')
            ->setParameter('type', Type::EMERGENCY);

        if ($unitID !== null) {
            $qb
                ->andWhere('u.uuid != :unitId')
                ->setParameter('unitId', $unitID, 'uuid');
        }

        if (!empty($unitChildrenUUIDs)) {
            $qb
                ->andWhere($qb->expr()->notIn('u.uuid', ':childrenUUIDs'))
                ->setParameter('childrenUUIDs', $unitChildrenUUIDs, Connection::PARAM_STR_ARRAY);
        }

        $unitsWithoutChildren = $qb
            ->execute()
            ->fetchAllAssociative();


        $results = array_merge($unitChildren, $unitsWithoutChildren);

        usort($results, function ($a, $b) {
        return strcasecmp($a["unitTitle"], $b["unitTitle"]);
        });

        return array_map(fn($result) => UnitChildrenDTO::FromArray($result), $results);

    }

    private function findAllUnitChildrenByUnitID(UnitID $unitID): array
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('u.title AS unitTitle, u.uuid AS unitId, uC.parent_unit_id AS parentUnitId')
            ->from('unit', 'u')
            ->leftJoin('u', 'unit_children', 'uC', 'uC.children_unit_id = u.uuid')
            ->where('uC.parent_unit_id = :unitId')
            ->andWhere('u.type = :type')
            ->setParameter('unitId', $unitID, 'uuid')
            ->setParameter('type', Type::EMERGENCY)
            ->execute()
            ->fetchAllAssociative();

        return $results;
    }

    public function collectedMoneyInUnitChildren(): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        return $qb
            ->select('u.uuid AS unit_id')
            ->addSelect('SUM(sO.collectedMoney) as collectedMoney')
            ->from('unit', 'u')
            ->leftJoin('u', 'unit_children', 'uC', 'uC.parent_unit_id = u.uuid')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('uC', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyInUnitChildrenQuery(), $qb), 'sO', 'uC.children_unit_id = sO.unit_id AND sO.orderDate >= p.started_at')
            ->groupBy('u.uuid');
    }

    private function unitHasParents(UnitID $unitID): bool
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('count(u.uuid)')
            ->from('unit', 'u')
            ->leftJoin('u', 'unit_children', 'uC', 'uC.parent_unit_id = u.uuid')
            ->where('uC.children_unit_id = :unitId')
            ->setParameter('unitId', $unitID, 'uuid')
            ->execute()
            ->fetchOne();

        return $results > 0 ;
    }

    private function unitIsEmergency(UnitID $unitID): bool
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('count(u.uuid)')
            ->from('unit', 'u')
            ->where('u.uuid = :unitId')
            ->andWhere('u.type = :type')
            ->setParameter('unitId', $unitID, 'uuid')
            ->setParameter('type', Type::EMERGENCY)
            ->execute()
            ->fetchOne();
        
        return $results > 0 ;
    }

    public function findAllUserUnitsToAssign(UserID $userID): array
    {
        $qb = $this->connection->createQueryBuilder();

        $results = $qb
            ->select('u.uuid AS unitId')
            ->addSelect('u.title AS unitTitle')
            ->addSelect('rP.user_id AS userId')
            ->from('unit', 'u')
            ->leftJoin('u', 'redactor_partners', 'rP', 'u.uuid = rP.unit_id AND (rP.user_id = :userId OR rP.user_id IS NULL)')
            ->setParameter('userId', $userID, 'uuid')
            ->orderBy('u.title', 'ASC')
            ->execute()
            ->fetchAllAssociative();

        return array_map(fn($result) => UnitToAssignListDTO::FromArray($result), $results);
    }
}