<?php

namespace App\Services\Orders\Webhooks;

use App\Core\Transactions\TransactionInterface;
use App\Entity\Orders\Classes\OrderManagerInterface;
use App\Entity\Orders\Events\OrderVerifyEvent;
use App\Entity\Orders\Order;
use App\PaymentSystem\Przelewy24Interface;
use App\PaymentSystem\TransactionVerification;
use Psr\EventDispatcher\EventDispatcherInterface;
use Psr\Log\LoggerInterface;

class VerifyOrder
{
    public function __construct(
        private TransactionInterface $transaction,
        private Przelewy24Interface $p24,
        private OrderManagerInterface $orderManager,
        private EventDispatcherInterface $eventDispatcher,
        private LoggerInterface $paymentLogger
    )
    {
    }

    public function verify(Order $order): Order
    {
        $webhook = $this->p24->handleWebhook();

        $this->paymentLogger->warning($webhook->statement());
        $this->paymentLogger->info($webhook->getTransactionNumber());

        $externalID = $webhook->orderId();
        $transactionNumber = $webhook->getTransactionNumber();
        try {

            $this->p24->verify(new TransactionVerification([
                    'sessionId' => (string)$order->getUuid(),
                    'orderId' => (int)$externalID,
                    'amount' => $order->getAmount(),
                ]
            ));

            $this->transaction->begin();

            $this->orderManager->orderVerified($order, $externalID, $transactionNumber);
            $this->eventDispatcher->dispatch(new OrderVerifyEvent($order), OrderVerifyEvent::NAME);

            $this->transaction->end();
        } catch (\Throwable $e
        ) {
            $this->paymentLogger->critical($e->getMessage());
        }

        return $order;
    }
}