<?php

namespace App\Services\Orders;

use App\Adapter\Subscriptions\Email\SubscriptionNotificationsInterface;
use App\Core\Transactions\TransactionInterface;
use App\Entity\Orders\Classes\OrderManagerInterface;
use App\Entity\Orders\OrdersInterface;
use App\Entity\Subscriptions\Helper\ReasonForClosure;
use App\Entity\Subscriptions\Subscription;
use App\Entity\Subscriptions\SubscriptionsInterface;
use App\PaymentSystem\Exceptions\ApiResponseException;
use App\PaymentSystem\PaymentTransaction;
use App\PaymentSystem\Przelewy24Interface;
use App\PaymentSystem\TransactionCardCharge;
use Psr\Log\LoggerInterface;

class SubscriptionCardCharge
{
    public function __construct(
        private TransactionInterface $transaction,
        private Przelewy24Interface $p24,
        private OrderManagerInterface $orderManager,
        private OrdersInterface $orders,
        private SubscriptionsInterface $subscriptions,
        private LoggerInterface $paymentLogger,
        private SubscriptionNotificationsInterface $subscriptionNotifications
    )
    {
    }

    public function createOrderForSubscriptionCardCharge(Subscription $subscription): string
    {

        $dateNow = new \DateTimeImmutable();

        $order = $this->orderManager->subscriptionCharge($subscription);

        $paymentTransaction = PaymentTransaction::fromOrder($order)
            ->addMethodRefId($subscription->getMethodRefID())
            ->addChannel(PaymentTransaction::CHANNEL_CARD);


        $paymentTransaction = $this->p24->transaction(
            $paymentTransaction
        );

        $this->paymentLogger->info('Registered order: ' . $order->getUuid() . ' for subscription card charge: ' . $subscription->getUuid() . ' at date: ' . $dateNow->format("Y-m-d H:i:s"));

        /** @var Subscription $subscription */
        $subscription = $order->getSubscription();


        $this->transaction->begin();

        try {

            $externalOrder = $this->p24->cardCharge(
                new TransactionCardCharge(
                    $paymentTransaction->token())
            );

            $order->setToken($paymentTransaction->token());
            $order->setExternalID($externalOrder->order());

            $subscription->cardChargeSuccess();

            $this->paymentLogger->info('Success payment order:' . $order->getUuid() . ' for subscription card charge: ' . $subscription->getUuid() . ' at date: ' . $dateNow->format("Y-m-d H:i:s"));

            $this->orders->add($order);
        } catch (ApiResponseException $e) {

            $subscription->cardChargeError();

            $this->subscriptionNotifications->sendCardChargeErrorEmail($subscription);

            $this->paymentLogger->error('Error payment order: ' . $order->getUuid() . ' for subscription card charge: ' . $subscription->getUuid() . ' at date: ' . $dateNow->format("Y-m-d H:i:s") . 'Count attempts for subscription: ' . $subscription->getAttempts());
            $this->paymentLogger->error('Error code: '.$e->getCode().', error message: '.$e->getMessage());

            if ($subscription->isMaxAttempts())
                $subscription->deactivate(ReasonForClosure::PAYMENTS);
        }


        $this->transaction->end();

        return $order->getUuid();
    }
}