<?php

namespace App\Entity\Reports\ReadModel;

use App\Core\Utils\FromArray;
use App\Core\Utils\Money;
use App\Entity\Subscriptions\ReadModel\KindOfSubscriptionTranslations;
use App\Entity\Subscriptions\ReadModel\ReasonForClosureTranslations;
use App\Entity\Subscriptions\SubscriptionID;
use App\Entity\Units\ReadModel\Helper\Type;
use App\Entity\Units\UnitID;
use App\Entity\Users\UserID;

class SubscriptionReportDTO
{
    use FromArray;

    public function __construct(
        private string $subscriptionId,
        private string $unitId,
        private string $unitTitle,
        private string $unitType,
        private string $userId,
        private string $userName,
        private string $userSurname,
        private string $userEmail,
        private string $kindOfSubscription,
        private string $createdAt,
        private int $monthlyCost,
        private int $children,
        private ?string $dateExpiration = null,
        private ?string $dateEnd = null,
        private ?string $reason = null,
        private ?string $referenceCode = null,
    )
    {
        $this->subscriptionId = (string)SubscriptionID::fromString($this->subscriptionId);
        $this->unitId = (string)UnitID::fromString($this->unitId);
        $this->userId = (string)UserID::fromString($this->userId);
    }

    /**
     * @return string
     */
    public function getSubscriptionID(): string
    {
        return $this->subscriptionId;
    }

    /**
     * @return string
     */
    public function getUnitID(): string
    {
        return $this->unitId;
    }

    /**
     * @return string
     */
    public function getUnitTitle(): string
    {
        return $this->unitTitle;
    }

    /**
     * @return string
     */
    public function getUnitType(): string
    {
        return (string)new Type($this->unitType);
    }

    /**
     * @return string
     */
    public function getUserID(): string
    {
        return $this->userId;
    }

    /**
     * @return string
     */
    public function getUserName(): string
    {
        return $this->userName;
    }

    /**
     * @return string
     */
    public function getUserSurname(): string
    {
        return $this->userSurname;
    }

    /**
     * @return string
     */
    public function getUserEmail(): string
    {
        return $this->userEmail;
    }

    /**
     * @return string|null
     */
    public function getReferenceCode(): ?string
    {
        return $this->referenceCode;
    }

    /**
     * @return string
     */
    public function getKindOfSubscription(): string
    {
        return (string)new KindOfSubscriptionTranslations($this->kindOfSubscription);
    }

    /**
     * @return string
     */
    public function getCreatedAt(): string
    {
        return $this->createdAt;
    }

    /**
     * @return string|null
     */
    public function getDateExpiration(): ?string
    {
        return $this->dateExpiration;
    }

    /**
     * @return string|null
     */
    public function getDateEnd(): ?string
    {
        return $this->dateEnd;
    }

    /**
     * @return string|null
     */
    public function getReason(): ?string
    {
        return $this->reason ? (string)new ReasonForClosureTranslations($this->reason) : null;
    }

    /**
     * @return float
     */
    public function getMoney(): float
    {
        return Money::toRational($this->monthlyCost * $this->children);
    }
}