<?php

namespace App\Entity\Purposes;

use Doctrine\ORM\Mapping as ORM;
use Symfony\Component\Uid\Uuid;
use App\Entity\Contents\Content;

/**
 * @ORM\Entity()
 */
class Purpose
{
    /**
     * @ORM\Id
     * @ORM\Column(type="uuid")
     */
    private PurposeID|Uuid $uuid;

    /**
     * @ORM\OneToOne(targetEntity=Content::class, cascade={"persist"})
     * @ORM\JoinColumn(name="content_id", referencedColumnName="uuid", nullable=false)
     */
    private Content $content;

    /**
     * @ORM\Column(type="string", length=255)
     */
    private string $title;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $createdAt;

    /**
     * @ORM\Column(type="datetime_immutable")
     */
    private \DateTimeImmutable $startedAt;

    /**
     * @ORM\Column(type="integer", options={"default": 0})
     */
    private int $money = 0;

    /**
     * @ORM\Column(type="datetime_immutable", nullable=true)
     */
    private ?\DateTimeImmutable $modifiedAt;

    /**
     * @param Content $content
     * @param string $title
     */
    public function __construct(Content $content, string $title)
    {
        $this->uuid = PurposeID::generate();
        $this->content = $content;
        $this->title = $title;
        $this->startedAt = new \DateTimeImmutable();
        $this->createdAt = new \DateTimeImmutable();
    }

    /**
     * @param string $title
     * @param int $money
     * @param \DateTimeImmutable $startedAt
     * @param string $content
     */
    public function update(string $title, int $money, \DateTimeImmutable $startedAt, string $content)
    {
        $this->title = $title;
        $this->money = $money;
        $this->startedAt = $startedAt;
        $this->content->updateContent($content);
        $this->modifiedAt = new \DateTimeImmutable();
    }

    /**
     * @return PurposeID
     */
    public function getUuid(): PurposeID
    {
        return PurposeID::fromString($this->uuid);
    }

    /**
     * @return Content
     */
    public function getContent(): Content
    {
        return $this->content;
    }

    public function getDescription(): string
    {
        return $this->getContent()->getContent();
    }



    /**
     * @param string $content
     */
    public function updateContent(string $content): void
    {
        $this->content->updateContent($content);
    }


    /**
     * @return string
     */
    public function getTitle(): string
    {
        return $this->title;
    }


    /**
     * @return \DateTimeImmutable
     */
    public function getCreatedAt(): \DateTimeImmutable
    {
        return $this->createdAt;
    }

    /**
     * @return \DateTimeImmutable
     */
    public function getStartedAt(): \DateTimeImmutable
    {
        return $this->startedAt;
    }

    /**
     * @return int
     */
    public function getMoney(): int
    {
        return $this->money;
    }

    /**
     * @param int $money
     */
    public function setMoney(int $money): void
    {
        $this->money = $money;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getModifiedAt(): ?\DateTimeImmutable
    {
        return $this->modifiedAt;
    }
}