<?php

namespace App\Entity\Orders\Classes;

use App\Core\Exception\ApiException;
use App\Core\UrlGenerator\Orders\OrderUrlGeneratorInterface;
use App\Entity\Orders\Helper\OrderAction;
use App\Entity\Orders\Helper\OrderAmount;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\Helper\RenewInformation;
use App\Entity\Orders\Order;
use App\Entity\Payments\Payment;
use App\Entity\Subscriptions\Helper\KindOfSubscription;
use App\Entity\Subscriptions\Subscription;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use App\Entity\Users\UsersInterface;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Contracts\Translation\TranslatorInterface;

class OrderManager implements OrderManagerInterface
{
    private ?User $volunteer = null;

    public function __construct(
        private OrderUrlGeneratorInterface $orderUrlGenerator,
        private RequestStack $requestStack,
        private UsersInterface $users,
        private TranslatorInterface $translator
    )
    {
        $token = $this->requestStack->getCurrentRequest()?->get('token');
        $this->volunteer = $token ? $this->users->findOneByToken($token) : null;
    }

    public function createPurposeOrder(string $email, int $amount, Unit $unit, ?User $user): Order
    {
        $order = $this->createOrder($email, $amount, $unit, $user, $unit->getPurpose()->getTitle());

        return $order
            ->setAction(OrderAction::PURPOSE)
            ->addPayment($this->createPayment($order, $amount));
    }

    public function createCyclicSubscriptionOrder(int $children, Unit $unit, User $user): Order
    {
        $amount = $this->getAmount($children, $unit->getHearthAdoption()->getMonthlyCost());

        $order = $this->createOrder($user->getEmail(), $amount->getMonthAmount(), $unit, $user, $unit->getType()->getDefaultDescription());

        $subscription = $this->createSubscription($user, $unit, $children, $order);
        $subscription->setKindOfSubscription(KindOfSubscription::CYCLIC);

        $order
            ->setAction(OrderAction::HEARTH_ADOPTION)
            ->setSubscription($subscription)
            ->addPayment($this->createPayment($order, $amount->getMonthAmount()));

        return $order;
    }

    /**
     * @throws \Exception
     */
    public function createOneTimeSubscriptionOrder(int $children, Unit $unit, User $user, int $months): Order
    {
        $amount = $this->getAmount($children, $unit->getHearthAdoption()->getMonthlyCost());


        $order = $this->createOrder($user->getEmail(), $amount->getTotalAmountByMonths($months), $unit, $user, $unit->getType()->getDefaultDescription());

        $subscription = $this->createSubscription($user, $unit, $children, $order);
        $subscription->setKindOfSubscription(KindOfSubscription::ONE_TIME);
        $subscription->setDateEndByMonths($months);
        $subscription->setDateExpirationByMonths($months);

        $order
            ->setAction(OrderAction::HEARTH_ADOPTION)
            ->setSubscription($subscription)
            ->setMonths($months);

        for ($i = 1; $i <= $months; $i++) {
            $simulatedMonths = $i - 1;

            $order->addPayment(
                $this->createPayment($order, $amount->getMonthAmount())
                    ->simulateCreatedAt($simulatedMonths)
            );
        }

        return $order;
    }

    public function createRenewSubscriptionOrder(Subscription $subscription, int $months, int $monthlyCost): Order
    {
        $this->setVolunteer($subscription->getVolunteer());

        $children = $subscription->getChildren();
        $user = $subscription->getUser();
        $unit = $subscription->getUnit();
        $amount = $this->getAmount($children, $monthlyCost);

        $order = $this->createOrder($user->getEmail(), $amount->getTotalAmountByMonths($months), $unit, $user, $unit->getType()->getDefaultDescription());
        $order
            ->setAction(OrderAction::HEARTH_ADOPTION)
            ->setSubscription($subscription)
            ->setUrlWebhook($this->orderUrlGenerator->generateRenewSubscriptionWebhookURL(['orderID' => $order->getUuid()]))
            ->setRenewInformation(new RenewInformation($children, $months, $monthlyCost))
            ->setMonths($months);


        return $order;
    }

    public function orderVerified(Order $order, string $externalID, string $transactionNumber): Order
    {
        return $order
            ->setVerifiedAt(new \DateTimeImmutable())
            ->setExternalID($externalID)
            ->setStatus(OrderStatus::VERIFIED)
            ->setTransactionNumber($transactionNumber);
    }


    public function subscriptionCharge(Subscription $subscription): Order
    {
        $this->setVolunteer($subscription->getVolunteer());

        $amount = $this->getAmount($subscription->getChildren(), $subscription->getMonthlyCost());

        $user = $subscription->getUser();
        $unit = $subscription->getUnit();

        $order = $this->createOrder($user->getEmail(), $amount->getMonthAmount(), $unit, $user, $unit->getType()->getDefaultDescription());

        $order
            ->setAction(OrderAction::HEARTH_ADOPTION)
            ->setSubscription($subscription)
            ->addPayment($this->createPayment($order, $amount->getMonthAmount()));

        return $order;
    }

    public function renewSubscription(Order $order): Order
    {
        $months = $order->getRenewInformation()->getMonths();
        $monthlyCost = $order->getRenewInformation()->getMonthlyCost();
        $children = $order->getRenewInformation()->getChildren();

        $amount = $this->getAmount($children, $monthlyCost);

        $subscription = $order->getSubscription();
        $subscription->renew($order);


        for ($i = 1; $i <= $months; $i++) {
            $simulatedMonths = $i - 1;

            $order->addPayment(
                $this->createPayment($order, $amount->getMonthAmount())->simulateCreatedAt($simulatedMonths)
            );
        }

        return $order;
    }

    /**
     * @param string $email
     * @param int $amount
     * @param Unit $unit
     * @param User|null $user
     * @param string $description
     * @return Order
     * @throws ApiException
     */
    private function createOrder(string $email, int $amount, Unit $unit, ?User $user, string $description): Order
    {
        if ($unit->isUnpublish())
            throw new ApiException($this->translator->trans('unit.unpublish'));

        $order = new Order($unit, $amount);

        return $order
            ->setEmail($email)
            ->setUnit($unit)
            ->setStatus(OrderStatus::WAITING_FOR_VERIFICATION)
            ->setUrlReturn($this->orderUrlGenerator->generateReturnURL(['orderID' => $order->getUuid()]))
            ->setUrlWebhook($this->orderUrlGenerator->generateWebhookURL(['orderID' => $order->getUuid()]))
            ->setUrlCardInfo($this->orderUrlGenerator->generateCardInfoURL(['orderID' => $order->getUuid()]))
            ->setUser($user)
            ->setDescription($description)
            ->setVolunteer($this->getVolunteer());
    }

    /**
     * @param User $user
     * @param Unit $unit
     * @param int $children
     * @return Subscription
     */
    private function createSubscription(User $user, Unit $unit, int $children, Order $basicOrder): Subscription
    {
        $subscription = new Subscription(
            $children,
            $unit->getHearthAdoption()->getMonthlyCost(),
            new \DateTimeImmutable('now'),
            $user,
            $unit,
            $basicOrder
        );

        $subscription->setVolunteer($this->getVolunteer());

        return $subscription;
    }

    /**
     * @param Order $order
     * @param int $amount
     * @return Payment
     */
    private function createPayment(Order $order, int $amount): Payment
    {
        return new Payment($order, $amount);
    }

    /**
     * @param int $children
     * @param int $monthlyCost
     * @return OrderAmount
     */
    private function getAmount(int $children, int $monthlyCost): OrderAmount
    {
        return new OrderAmount($children, $monthlyCost);
    }

    /**
     * @param User|null $volunteer
     * @return OrderManager
     */
    public function setVolunteer(?User $volunteer): OrderManager
    {
        $this->volunteer = $volunteer;
        return $this;
    }

    /**
     * @return User|null
     */
    public function getVolunteer(): ?User
    {
        return $this->volunteer;
    }
}