<?php

namespace App\Controller\Units\Update;

use App\Entity\Permissions\Permission;
use App\Entity\Units\Unit;
use App\Models\Units\UnitModel;
use App\Services\Units\UpdateUnit;
use App\Services\Units\UpdateUnitStatus;
use OpenApi\Annotations as OA;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Units")
 */
#[Route(path: '/api/units', name: 'units_')]
class UpdateUnitController extends AbstractController
{
    public function __construct(
        private UpdateUnit $updateUnit,
        private UpdateUnitStatus $unitStatus
    ) {}

    /**
     * @OA\Post(summary="Update unit")
     * @OA\RequestBody(
     *     @OA\MediaType(
     *          mediaType="multipart/form-data",
     *          @OA\Schema(
     *              type="object",
     *              required={"title", "description"},
     *              @OA\Property(property="title", type="string"),
     *              @OA\Property(property="description", type="string"),
     *              @OA\Property(property="photo", type="file"),
     *              @OA\Property(property="banner", type="file"),
     *              @OA\Property(property="unitChildren", type="string")
     *          )
     *     )
     * )
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(path: '/{unitId}', name: 'update', methods: ['POST'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitId' => 'uuid']])]
    #[ParamConverter('unitModel', options: ["groups" => [UnitModel::UPDATE]], converter: 'form_data_deserialize_param_converter')]
    public function update(Unit $unit, UnitModel $unitModel): JsonResponse
    {
        $this->updateUnit->update($unit, $unitModel);
        return $this->json('Unit has been updated');
    }

    /**
     * @OA\Put(summary="Publish unit")
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(path: '/{unitId}/publish', name: 'publish_status', methods: ['PUT'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitId' => 'uuid']])]
    public function published(Unit $unit): JsonResponse
    {
        $this->unitStatus->publish($unit);
        return $this->json('Unit status has been updated');
    }

    /**
     * @OA\Put(summary="Unpublish unit")
     */
    #[IsGranted(Permission::UNIT_ADMINISTRATION)]
    #[Route(path: '/{unitId}/unpublish', name: 'unpublish_status', methods: ['PUT'])]
    #[ParamConverter('unit', options: ['mapping' => ['unitId' => 'uuid']])]
    public function unpublished(Unit $unit): JsonResponse
    {
        $this->unitStatus->unpublish($unit);
        return $this->json('Unit status has been updated');
    }
}