<?php

namespace App\Controller\Orders;

use App\Entity\Permissions\Permission;
use App\Entity\Subscriptions\Subscription;
use App\Models\Orders\RenewSubscriptionModel;
use App\Services\Orders\SubscriptionCancel;
use App\Services\Orders\SubscriptionCardCharge;
use App\Services\Orders\SubscriptionRenew;
use OpenApi\Annotations as OA;
use Nelmio\ApiDocBundle\Annotation\Model;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\IsGranted;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

/**
 * @OA\Tag(name="Orders")
 */
#[Route(path: '/api/orders', name: 'orders_')]
class SubscriptionController extends AbstractController
{

    // TODO endpoint to check after end coding
    /**
     * @OA\Get(summary="Manually charge card by cyclic subscription")
     * @param Subscription $subscription
     * @param SubscriptionCardCharge $createOrderForSubscription
     * @return Response
     */
    #[IsGranted(Permission::PAYMENTS_MANAGEMENT)]
    #[Route('/{subscriptionID}/card/charge', name: 'card_charge', methods: ['GET'])]
    #[ParamConverter('subscription', options: ['mapping' => ['subscriptionID' => 'uuid']])]
    public function cardCharge(Subscription $subscription, SubscriptionCardCharge $createOrderForSubscription): Response
    {
        $this->denyAccessUnlessGranted(Permission::PRIVATE_ACCESS, $subscription->getUser());

        $orderID = $createOrderForSubscription->createOrderForSubscriptionCardCharge($subscription);

        return $this->json(['Success' => $orderID]);
    }

    /**
     * @OA\Post(summary="Cancel subscription")
     * @param Subscription $subscription
     * @param SubscriptionCancel $subscriptionCancel
     * @return Response
     */
    #[IsGranted(Permission::PAYMENTS_MANAGEMENT)]
    #[Route('/{subscriptionID}/cancel', name: 'subscription_cancel', methods: ['POST'])]
    #[ParamConverter('subscription', options: ['mapping' => ['subscriptionID' => 'uuid']])]
    public function cancel(Subscription $subscription, SubscriptionCancel $subscriptionCancel): Response
    {
        $this->denyAccessUnlessGranted(Permission::PRIVATE_ACCESS, $subscription->getUser());

        $subscriptionCancel->cancel($subscription);

        return $this->json('Subscription successfully canceled');
    }

    /**
     * @OA\RequestBody(
     *     @Model(type=RenewSubscriptionModel::class)
     * )
     * @OA\Response(response=200, description="Return url to redirect")
     * @param Subscription $subscription
     * @param RenewSubscriptionModel $renewSubscriptionModel
     * @param SubscriptionRenew $subscriptionRenew
     * @return Response
     */
    #[IsGranted(Permission::PAYMENTS_MANAGEMENT)]
    #[Route('/{subscriptionID}/renew', name: 'renew', methods: ['POST'])]
    #[ParamConverter('subscription', options: ['mapping' => ['subscriptionID' => 'uuid']])]
    public function renew(Subscription $subscription, RenewSubscriptionModel $renewSubscriptionModel, SubscriptionRenew $subscriptionRenew): Response
    {
        $this->denyAccessUnlessGranted(Permission::PRIVATE_ACCESS, $subscription->getUser());

        $redirectUrl = $subscriptionRenew->renewOneTimeSubscription($subscription, $renewSubscriptionModel);

        return $this->json(['redirectURL' => $redirectUrl]);
    }
}