<?php

namespace App\Command;

use App\Entity\Orders\OrderID;
use App\Entity\Orders\OrdersInterface;
use App\Services\Orders\UpdateTransactionNumber;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(
    name: 'orders:transaction-number:update',
    description: 'Update order transaction number',
)]
class OrdersTransactionNumberCommand extends Command
{

    public function __construct(
        private OrdersInterface $orders,
        private UpdateTransactionNumber $updateTransactionNumber
    )
    {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->addArgument('limit', InputArgument::OPTIONAL, 'Limit of orders')
            ->addArgument('orderId', InputArgument::OPTIONAL, 'Order ID')
            ->addOption('id', 'i', InputOption::VALUE_NONE, 'Option description');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);
        $orderId = null;
        $limit = null;

        if ($input->getOption('id')) {
            if (!$input->hasArgument('orderId')) {
                $io->error('Order id not provided');
                return Command::FAILURE;
            }
            $orderId = OrderID::fromString($input->getArgument('orderId'));
        }

        if ($input->hasArgument('limit')) {
            $limit = (int)$input->getArgument('limit');
        }
        $orders = $this->orders->findWithoutTransactionNumber($orderId, $limit);
        $count = count($orders);
        $countErrors = 0;

        foreach ($orders as $order) {
            try {
                $this->updateTransactionNumber->update($order);
            } catch (\Throwable $e) {
                $countErrors++;
                $io->error($e->getMessage());
            }
        }
        $io->success(sprintf('Command end with success, count of orders: %s, count of errors %s', $count, $countErrors));

        return Command::SUCCESS;
    }
}
