<?php

namespace App\Adapter\Subscriptions;

use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\Order;
use App\Entity\Subscriptions\Helper\KindOfSubscription;
use App\Entity\Subscriptions\Helper\RemindDays;
use App\Entity\Subscriptions\Subscription;
use App\Entity\Subscriptions\SubscriptionsInterface;
use App\Entity\Units\Unit;
use App\Entity\Users\User;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Query\Expr\Join;
use function Doctrine\ORM\QueryBuilder;

class Subscriptions implements SubscriptionsInterface
{
    public function __construct(
        private EntityManagerInterface $entityManager
    )
    {
    }

    /**
     * Find all subscriptions to daily charge
     * @param \DateTimeImmutable|null $dateTime
     * @return array
     */
    public function findAllToCharge(?\DateTimeImmutable $dateTime = null): array
    {
        $date = $dateTime ?: new \DateTimeImmutable();
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->select('s')
            ->from(Subscription::class, 's')
            ->leftJoin(Order::class, 'o', Join::WITH, 's.basicOrderID = o.uuid')
            ->where('s.nextCardCharge = :dateNow')
            ->andWhere('s.isActive = :isActive')
            ->andWhere('o.status = :orderStatus')
            ->andWhere('s.kindOfSubscription = :kindOfSubscription')
            ->setParameter('dateNow', $date->format('Y-m-d'))
            ->setParameter('isActive', true)
            ->setParameter('kindOfSubscription', KindOfSubscription::CYCLIC)
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->getQuery()
            ->getResult();
    }

    /**
     * Find all subscriptions with dateEnd = currentDate or $dateTime
     * @param \DateTimeImmutable|null $dateTime
     * @return array
     */
    public function findAllExpired(?\DateTimeImmutable $dateTime = null): array
    {
        $date = $dateTime ?: new \DateTimeImmutable();
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->select('s')
            ->from(Subscription::class, 's')
            ->where('DATE(s.dateEnd) = :dateNow')
            ->andWhere('s.isActive = :isActive')
            ->setParameter('dateNow', $date->format('Y-m-d'))
            ->setParameter('isActive', true)
            ->getQuery()
            ->getResult();
    }

    /**
     * Find all subscriptions with basic order status unverified
     * @return Subscription[]
     */
    public function findAllUnpaid(): array
    {
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->select('s')
            ->from(Subscription::class, 's')
            ->leftJoin(Order::class, 'o', Join::WITH, 's.basicOrderID = o.uuid')
            ->where('o.status != :orderStatus')
            ->andWhere('s.isActive = :isActive')
            ->setParameter('isActive', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->getQuery()
            ->getResult();
    }

    public function findAllActiveByUser(User $user): array
    {
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->select('s')
            ->from(Subscription::class, 's')
            ->leftJoin(Order::class, 'o', Join::WITH, 's.basicOrderID = o.uuid')
            ->where('o.status = :orderStatus')
            ->andWhere('s.isActive = :isActive')
            ->andWhere('s.user = :user')
            ->setParameter('user', $user->getUuid(), 'uuid')
            ->setParameter('isActive', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->getQuery()
            ->getResult();
    }

    public function findAllActiveByUnit(Unit $unit): array
    {
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->select('s')
            ->from(Subscription::class, 's')
            ->leftJoin(Order::class, 'o', Join::WITH, 's.basicOrderID = o.uuid')
            ->where('o.status = :orderStatus')
            ->andWhere('s.isActive = :isActive')
            ->andWhere('s.unit = :unit')
            ->setParameter('unit', $unit->getUuid(), 'uuid')
            ->setParameter('isActive', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->getQuery()
            ->getResult();
    }

    public function findAllToRemind(): array
    {
        $qb = $this->entityManager->createQueryBuilder();

        return $qb
            ->addSelect('s')
            ->from(Subscription::class, 's')
            ->leftJoin(Unit::class, 'u', Join::WITH, 's.unit = u.uuid')
            ->leftJoin(User::class, 'us', Join::WITH, 's.user = us.uuid')
            ->leftJoin(Order::class, 'o', Join::WITH, 's.basicOrderID = o.uuid')
            ->andWhere('o.status = :orderStatus')
            ->andWhere('s.isActive = :isActive')
            ->andWhere($qb->expr()->in('DATEDIFF(s.dateEnd, :dateNow)', RemindDays::ALL_DAYS))
            ->setParameter('dateNow', (new \DateTimeImmutable())->format('Y-m-d H:i:s'))
            ->setParameter('isActive', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED)
            ->getQuery()
            ->getResult();
    }
}