<?php

namespace App\Adapter\Purposes\ReadModel;

use App\Core\Doctrine\SubQueryBuilder;
use App\Entity\Orders\ReadModel\OrdersSubQueryInterface;
use App\Entity\Purposes\PurposeID;
use App\Entity\Purposes\ReadModel\PurposeDTO;
use App\Entity\Purposes\ReadModel\GoalsListDTO;
use App\Entity\Purposes\ReadModel\PurposesQueryInterface;
use App\Entity\Units\UnitID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Query\QueryBuilder;

class PurposesQuery implements PurposesQueryInterface
{
    public function __construct(
        private Connection $connection,
        private OrdersSubQueryInterface $ordersSubQuery
    ) {}

    /**
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findGoalsListByUnitId(UnitID $unitID): ?GoalsListDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('u.type')
            ->addSelect('p.uuid as purposeId')
            ->addSelect('p.title as purposeTitle')
            ->addSelect('cP.content as purposeContent')
            ->addSelect('a.uuid as hearthAdoptionId')
            ->addSelect('p.money as moneyPurpose')
            ->addSelect('(COALESCE(sO.collectedMoney,0) + COALESCE(uC.collectedMoney,0)) as collectedMoney')
            ->addSelect('a.children')
            ->addSelect('a.monthly_cost as monthlyCost')
            ->addSelect('s.supportedChildren as supportedChildren')
            ->addSelect('cA.content as adoptionContent')
            ->from('unit', 'u')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('p', 'content', 'cP', 'p.content_id = cP.uuid')
            ->leftJoin('u', 'hearth_adoption', 'a', 'u.adoption_id = a.uuid')
            ->leftJoin('a', 'content', 'cA', 'a.content_id = cA.uuid')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyQuery(), $qb), 'sO', 'u.uuid = sO.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->ordersSubQuery->supportedChildrenQuery(), $qb), 's', 'u.uuid = s.unit_id')
            ->leftJoin('u', SubQueryBuilder::prepare($this->collectedMoneyInUnitChildren(), $qb), 'uC', 'u.uuid = uC.unit_id')
            ->where('u.uuid = :unitId')
            ->setParameter('unitId', $unitID, 'uuid')
            ->execute()
            ->fetch();

        return $result ? GoalsListDTO::fromArray($result) : null;
    }

    /**
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findOnePurposeById(PurposeID $purposeID): ?PurposeDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('p.title')
            ->addSelect('p.money')
            ->addSelect('p.started_at as startedAt')
            ->addSelect('c.content')
            ->from('unit', 'u')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('p', 'content', 'c', 'p.content_id = c.uuid')
            ->where('u.purpose_id = :purposeID')
            ->setParameter('purposeID', $purposeID, 'uuid')
            ->execute()
            ->fetch();

        return $result ? PurposeDTO::fromArray($result) : null;
    }

    public function collectedMoneyInUnitChildren(): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        return $qb
            ->select('u.uuid AS unit_id')
            ->addSelect('SUM(sO.collectedMoney) as collectedMoney')
            ->from('unit', 'u')
            ->leftJoin('u', 'unit_children', 'uC', 'uC.parent_unit_id = u.uuid')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->leftJoin('uC', SubQueryBuilder::prepare($this->ordersSubQuery->collectedMoneyInUnitChildrenQuery(), $qb), 'sO', 'uC.children_unit_id = sO.unit_id AND sO.orderDate >= p.started_at')
            ->groupBy('u.uuid');
    }
}