<?php

namespace App\Adapter\Posts\ReadModel;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\Posts\PostID;
use App\Entity\Posts\ReadModel\PostDataDTO;
use App\Entity\Posts\ReadModel\PostsQueryInterface;
use App\Entity\Units\UnitID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Driver\Exception as DriverException;

class PostsQuery implements PostsQueryInterface
{
    public function __construct(
        private Connection $connection,
        private PaginatorInterface $paginator
    ) {}

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllPostsByUnitId(UnitID $unitID, PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('p.uuid as postId')
            ->addSelect('p.title')
            ->addSelect('p.description')
            ->addSelect('p.created_at as createdAt')
            ->addSelect('i.image as photo')
            ->addSelect('p.is_published as isPublished')
            ->from('post', 'p')
            ->leftJoin('p', 'image', 'i', 'p.photo_id = i.uuid')
            ->where('p.unit_id = :unitId')
            ->andWhere('p.is_active = :postStatus')
            ->setParameter('unitId', $unitID, 'uuid')
            ->setParameter('postStatus', true)
            ->orderBy('p.created_at', 'DESC');

        $this->paginator->addPagination($query, $paginationRequest);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => PostDataDTO::fromArray($data), $results);
    }

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findPublishedPostsByUnitId(UnitID $unitID, PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('p.uuid as postId')
            ->addSelect('p.title')
            ->addSelect('p.description')
            ->addSelect('p.created_at as createdAt')
            ->addSelect('i.image as photo')
            ->addSelect('p.is_published as isPublished')
            ->from('post', 'p')
            ->leftJoin('p', 'image', 'i', 'p.photo_id = i.uuid')
            ->where('p.unit_id = :unitId')
            ->andWhere('p.is_active = :postStatus')
            ->andWhere('p.is_published = :publishedStatus')
            ->setParameter('unitId', $unitID, 'uuid')
            ->setParameter('postStatus', true)
            ->setParameter('publishedStatus', true)
            ->orderBy('p.created_at', 'DESC');

        $this->paginator->addPagination($query, $paginationRequest);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => PostDataDTO::fromArray($data), $results);
    }

    /**
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findOnePostById(PostID $postID): ?PostDataDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('p.uuid as postId')
            ->addSelect('p.title')
            ->addSelect('p.description')
            ->addSelect('c.content')
            ->addSelect('p.created_at as createdAt')
            ->addSelect('i.image as photo')
            ->addSelect('p.is_published as isPublished')
            ->from('post', 'p')
            ->leftJoin('p', 'content', 'c', 'p.content_id = c.uuid')
            ->leftJoin('p', 'image', 'i', 'p.photo_id = i.uuid')
            ->where('p.uuid = :postId')
            ->andWhere('p.is_active = :postStatus')
            ->setParameter('postId', $postID, 'uuid')
            ->setParameter('postStatus', true)
            ->execute()
            ->fetch();

        return $result ? PostDataDTO::fromArray($result) : null;
    }
}