<?php

namespace App\Adapter\Orders\ReadModel;

use App\Entity\Orders\Helper\OrderAction;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\ReadModel\OrdersSubQueryInterface;
use App\Entity\Users\UserID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Query\QueryBuilder;

class OrdersSubQuery implements OrdersSubQueryInterface
{
    public function __construct(
        private Connection $connection
    )
    {
    }

    public function collectedMoneyQuery(): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        return $qb
            ->select('sO.unit_id')
            ->addSelect('SUM(sO.amount) as collectedMoney')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'u.uuid = sO.unit_id')
            ->leftJoin('u', 'purpose', 'p', 'u.purpose_id = p.uuid')
            ->where('sO.action = :orderAction')
            ->andWhere('sO.status = :orderStatus')
            ->andWhere('sO.verified_at >= p.started_at')
            ->groupBy('sO.unit_id')
            ->setParameter('orderAction', OrderAction::PURPOSE)
            ->setParameter('orderStatus', OrderStatus::VERIFIED);
    }

    public function collectedMoneyInUnitChildrenQuery(): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        return $qb
            ->select('sO.unit_id')
            ->addSelect('SUM(sO.amount) as collectedMoney')
            ->addSelect('sO.verified_at as orderDate')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'u.uuid = sO.unit_id')
            ->where('sO.action = :orderAction')
            ->andWhere('sO.status = :orderStatus')
            ->groupBy('sO.unit_id')
            ->addGroupBy('sO.verified_at')
            ->setParameter('orderAction', OrderAction::PURPOSE)
            ->setParameter('orderStatus', OrderStatus::VERIFIED);
    }

    public function supportedChildrenQuery(): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        return $qb
            ->select('s.unit_id')
            ->addSelect('SUM(s.children) as supportedChildren')
            ->addSelect('SUM(s.children * s.monthly_cost) as supportedChildrenMoney')
            ->from('subscription', 's')
            ->leftJoin('s', 'sysOrder', 'bO', 's.basic_order_id = bO.uuid')
            ->where('s.is_active = :subscriptionStatus')
            ->andWhere('bO.status = :orderStatus')
            ->groupBy('s.unit_id')
            ->setParameter('subscriptionStatus', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED);
    }

    public function ordersAmount(?\DateTimeImmutable $dateStart = null, ?\DateTimeImmutable $dateEnd = null): QueryBuilder
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('sO.unit_id')
            ->addSelect('SUM(sO.amount) as totalMoney')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'u.uuid = sO.unit_id')
            ->andWhere('sO.status = :orderStatus')
            ->groupBy('sO.unit_id')
            ->setParameter('orderStatus', OrderStatus::VERIFIED);

        if ($dateStart)
            $this->addDateStartFilter($query, $dateStart);

        if ($dateEnd)
            $this->addDateEndFilter($query, $dateEnd);


        return $query;
    }

    public function ordersAmountByUserAndYear(UserID $userID, ?int $year = null): QueryBuilder
    {
        $query = $this->ordersAmount()
            ->andWhere('sO.user_id = :userID')
            ->setParameter('userID', $userID, 'uuid');

        if ($year) {
            $query
                ->andWhere('YEAR(sO.verified_at) = :year')
                ->setParameter('year', $year);
        }
        return $query;
    }

    private function addDateStartFilter(QueryBuilder $queryBuilder, \DateTimeImmutable $date)
    {
        $queryBuilder
            ->andWhere('DATE(sO.verified_at) >= :dateStart')
            ->setParameter('dateStart', $date->format('Y-m-d'));
    }

    private function addDateEndFilter(QueryBuilder $queryBuilder, \DateTimeImmutable $date)
    {
        $queryBuilder
            ->andWhere('DATE(sO.verified_at) <= :dateEnd')
            ->setParameter('dateEnd', $date->format('Y-m-d'));
    }

}