<?php

namespace App\Adapter\Orders\ReadModel;

use App\Core\Paginator\ObjectValue\PaginationRequest;
use App\Core\Paginator\PaginatorInterface;
use App\Entity\Orders\Helper\OrderStatus;
use App\Entity\Orders\OrderID;
use App\Entity\Orders\ReadModel\OrderDTO;
use App\Entity\Orders\ReadModel\OrdersQueryInterface;
use App\Entity\Orders\ReadModel\PaymentHistoryDTO;
use App\Entity\Orders\ReadModel\PaymentSubscriptionDTO;
use App\Entity\Users\UserID;
use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Exception;
use Doctrine\DBAL\Driver\Exception as DriverException;

class OrdersQuery implements OrdersQueryInterface
{
    public function __construct(
        private Connection $connection,
        private PaginatorInterface $paginator
    ) {}

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllPaymentByUserId(UserID $userID, PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('u.title as unitTitle')
            ->addSelect('sO.description as purpose')
            ->addSelect('sO.action')
            ->addSelect('sO.amount')
            ->addSelect('sO.verified_at as date')
            ->from('sysOrder', 'sO')
            ->leftJoin('sO', 'unit', 'u', 'sO.unit_id = u.uuid')
            ->where('sO.user_id = :userId')
            ->andWhere('sO.status = :status')
            ->orderBy('sO.verified_at', 'DESC')
            ->setParameter('userId', $userID, 'uuid')
            ->setParameter('status', OrderStatus::VERIFIED);

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => PaymentHistoryDTO::fromArray($data), $results);
    }

    /**
     * @throws Exception
     * @throws DriverException
     * @throws \ReflectionException
     */
    public function findAllSubscriptionByUserId(UserID $userID, PaginationRequest $paginationRequest): array
    {
        $qb = $this->connection->createQueryBuilder();

        $query = $qb
            ->select('s.uuid as subscriptionId')
            ->addSelect('s.date_start as dateStart')
            ->addSelect('u.title as unitTitle')
            ->addSelect('u.type as unitType')
            ->addSelect('s.children')
            ->addSelect('s.monthly_cost as monthlyCost')
            ->addSelect('s.date_end as dateEnd')
            ->addSelect('s.kind_of_subscription as kindOfSubscription')
            ->from('subscription', 's')
            ->leftJoin('s','sysOrder', 'o', 's.basic_order_id = o.uuid')
            ->leftJoin('s', 'unit', 'u', 's.unit_id = u.uuid')
            ->where('s.user_id = :userId')
            ->andWhere('s.is_active = :status')
            ->andWhere('o.status = :orderStatus')
            ->orderBy('s.date_start', 'DESC')
            ->setParameter('userId', $userID, 'uuid')
            ->setParameter('status', true)
            ->setParameter('orderStatus', OrderStatus::VERIFIED);
        //ToDo Add fake subscriptions left join

        $this->paginator->addPagination($query, $paginationRequest, true);
        $results = $query->execute()->fetchAllAssociative();

        return array_map(fn($data) => PaymentSubscriptionDTO::fromArray($data), $results);
    }

    /**
     * @param OrderID $orderID
     * @return OrderDTO|null
     * @throws Exception
     * @throws \ReflectionException
     */
    public function findOneByOrderID(OrderID $orderID): ?OrderDTO
    {
        $qb = $this->connection->createQueryBuilder();

        $result = $qb
            ->select('o.uuid as orderId')
            ->addSelect('o.created_at as createdAt')
            ->addSelect('o.status')
            ->addSelect('o.action')
            ->addSelect('o.amount')
            ->addSelect('u.title as unitTitle')
            ->addSelect('u.uuid as unitId')
            ->from('sysOrder', 'o')
            ->leftJoin('o', 'unit', 'u', 'o.unit_id = u.uuid')
            ->where('o.uuid = :orderID')
            ->setParameter('orderID', $orderID, 'uuid')
            ->execute()
            ->fetch();

        return $result ? OrderDTO::fromArray($result) : null;
    }
}